DATA height_long;
     INFILE 'C:\GRE2016\Data\bgs_height_long.dat';
     INPUT id age hght;
RUN;

*Jenss-Bayley Growth Model;

PROC NLMIXED DATA = height_long QPOINTS = 5;
     traject = b_1i + b_2i * age/12 + b_3i * (exp(b_4i*(age/12))-1);

     MODEL hght ~ NORMAL(traject, v_u);
     RANDOM b_1i b_2i b_3i b_4i ~ NORMAL([beta_1,beta_2,beta_3, beta_4],
                                         [v_1,
                                         c_21,  v_2,
                                         c_31, c_32,  v_3,
                                         c_41, c_42, c_43, v_4])
     SUBJECT = id 
     OUT = jb_estimates;

     PARMS
        beta_1 = 51   beta_2 = 9   beta_3 = -18   beta_4 = -2
        v_1 = 10      v_2 = .8     v_3 = 12       v_4 = .01
        c_21 = 0      c_31 = -4    c_32 = 1       c_41 = 0     c_42 = 0     c_43 = 0
        v_u = .7;
RUN;

/*
                                        Fit Statistics

                           -2 Log Likelihood                 1974.0
                           AIC (smaller is better)           2004.0
                           AICC (smaller is better)          2004.9
                           BIC (smaller is better)           2040.3


                                     Parameter Estimates

                      Standard
 Parameter  Estimate     Error    DF  t Value  Pr > |t|   Alpha     Lower     Upper  Gradient

 beta_1      51.0437    0.3509    79   145.46    <.0001    0.05   50.3452   51.7421  0.000086
 beta_2       9.2391    0.1760    79    52.48    <.0001    0.05    8.8887    9.5894  0.000224
 beta_3     -18.0513    0.5043    79   -35.79    <.0001    0.05  -19.0551  -17.0475  -0.00052
 beta_4      -2.1048   0.09031    79   -23.30    <.0001    0.05   -2.2845   -1.9250  -0.00274
 v_1          7.3250    1.4821    79     4.94    <.0001    0.05    4.3749   10.2751   0.00009
 v_2          0.8111    0.3007    79     2.70    0.0085    0.05    0.2125    1.4096  -0.00035
 v_3         11.6634    2.9227    79     3.99    0.0001    0.05    5.8459   17.4808   -4.5E-6
 v_4         0.08974   0.05363    79     1.67    0.0982    0.05  -0.01700    0.1965  -0.00468
 c_21         0.6796    0.4735    79     1.44    0.1551    0.05   -0.2628    1.6220  -5.12E-6
 c_31         4.5668    1.5268    79     2.99    0.0037    0.05    1.5278    7.6057  -0.00018
 c_32         1.3368    0.7735    79     1.73    0.0878    0.05   -0.2028    2.8764   0.00041
 c_41       -0.01599    0.2562    79    -0.06    0.9504    0.05   -0.5259    0.4939  -0.00085
 c_42       -0.09637    0.1182    79    -0.82    0.4174    0.05   -0.3317    0.1389  -0.00067
 c_43        -0.5482    0.3902    79    -1.41    0.1639    0.05   -1.3248    0.2284  0.000916
 v_u          0.6397   0.05027    79    12.73    <.0001    0.05    0.5397    0.7398  -0.00212
*/


*Predicted Plot based on OUT=estimate;
PROC SORT DATA = jb_estimates;
	BY id;
RUN;
DATA jb_estimates_b1;
     SET jb_estimates;
     IF effect = 'b_1i';
     b_1i = estimate;
     KEEP id b_1i;
RUN;
DATA jb_estimates_b2;
     SET jb_estimates;
     IF effect = 'b_2i';
     b_2i = estimate;
     KEEP id b_2i;
RUN;
DATA jb_estimates_b3;
     SET jb_estimates;
     IF effect = 'b_3i';
     b_3i = estimate;
     KEEP id b_3i;
RUN;
DATA jb_estimates_b4;
     SET jb_estimates;
     IF effect = 'b_4i';
     b_4i = estimate;
     KEEP id b_4i;
RUN;
DATA jb_estimates1;
     MERGE jb_estimates_b1 
           jb_estimates_b2
           jb_estimates_b3
           jb_estimates_b4;
     BY id;
RUN;
 
*Creating a longitudinal database with predictions;
DATA igs_prediction;
     MERGE height_long jb_estimates1;
     BY id;
     pred = b_1i + b_2i * age/12 + b_3i * (exp(b_4i*(age/12))-1);
     resid = hght - pred;
RUN;
GOPTIONS
     FTEXT = ARIAL
     CTEXT = BLACK
     HTEXT = 1.5;

PROC GPLOT DATA = igs_prediction;
     TITLE1 'IGS Height by Age - Predictions';
     SYMBOL
          Interpol=smooth
          Color=black
          Repeat=5000
          Value=dot
          Height=.5
          Width=1;
     AXIS1
          LABEL = (F=ARIAL H = 2 A=90 'Height (cm)')
          ORDER = (40 to 110 BY 10)
          OFFSET= (5 PCT, 5 PCT)
          MINOR = none;
     AXIS2
          LABEL = (F=ARIAL H = 2 'Age in Months')	
          ORDER = (0 to 40 BY 10)
          OFFSET= (5 PCT, 5 PCT)
          MINOR = none;

     PLOT pred * age = id/NOLEGEND VAXIS= AXIS1 HAXIS=AXIS2;
RUN;

DATA igs_prediction;
	SET jb_estimates1;
	DO age = 1 to 36 by .5;
	pred = b_1i + b_2i * age/12 + b_3i * (exp(b_4i*(age/12))-1);
	OUTPUT;
	END;
RUN;


PROC GPLOT DATA = igs_prediction;
	TITLE1 'IGS Height by Age - Predictions';
	SYMBOL
		Interpol=join
		Color=black
		Repeat=5000
		Value=none
		Height=.5
		Width=1;
	AXIS1
		LABEL = (F=ARIAL H = 2 A=90 'Height (cm)')
		ORDER = (40 to 110 BY 10)
		MINOR = none;
	AXIS2
		LABEL = (F=ARIAL H = 2 'Age in Months')	
		ORDER = (0 to 40 BY 10)
		OFFSET= (5 PCT, 5 PCT)
		MINOR = none;

	PLOT pred * age = id/NOLEGEND VAXIS= AXIS1 HAXIS=AXIS2;
RUN;



* Method = FIRO for first-order approximation;
PROC NLMIXED DATA = height_long METHOD=FIRO;
	traject = b_1i + b_2i * age/12 + b_3i * (exp(b_4i*(age/12))-1);

	MODEL hght ~ NORMAL(traject, v_u);
	RANDOM b_1i b_2i b_3i b_4i ~ NORMAL([beta_1,beta_2,beta_3, beta_4], [v_1,
									                            c_21,  v_2,
                                                                c_31, c_32,  v_3,
                                                                c_41, c_42, c_43, v_4])
	SUBJECT = id;

	PARMS
        beta_1 = 51   beta_2 = 9     beta_3 = -18   beta_4 = -2
        v_1 = 10    v_2 = .8     v_3 = 12     v_4 = .01
        c_21 = 0    c_31 = -4    c_32 = 1     c_41 = 0     c_42 = 0     c_43 = 0
        v_u = .7;
RUN;

/*
                                        Fit Statistics

                           -2 Log Likelihood                 1975.0
                           AIC (smaller is better)           2005.0
                           AICC (smaller is better)          2005.8
                           BIC (smaller is better)           2041.3

                                     Parameter Estimates

                      Standard
 Parameter  Estimate     Error    DF  t Value  Pr > |t|   Alpha     Lower     Upper  Gradient

 beta_1      51.0742    0.3520    79   145.11    <.0001    0.05   50.3736   51.7747  0.000036
 beta_2       9.2960    0.1738    79    53.49    <.0001    0.05    8.9501    9.6419  0.000054
 beta_3     -17.8366    0.4908    79   -36.34    <.0001    0.05  -18.8134  -16.8597  -0.00002
 beta_4      -2.0769   0.08448    79   -24.58    <.0001    0.05   -2.2450   -1.9087   0.00013
 v_1          7.4107    1.5159    79     4.89    <.0001    0.05    4.3934   10.4280  -8.75E-6
 v_2          0.8090    0.3375    79     2.40    0.0189    0.05    0.1372    1.4808  -0.00007
 v_3         11.6224    2.9984    79     3.88    0.0002    0.05    5.6544   17.5905  -0.00001
 v_4         0.07590   0.07833    79     0.97    0.3355    0.05  -0.08001    0.2318  -1.79E-6
 c_21         0.6390    0.5275    79     1.21    0.2294    0.05   -0.4111    1.6890  0.000047
 c_31         4.4487    1.5728    79     2.83    0.0059    0.05    1.3181    7.5794  0.000039
 c_32         1.3444    0.8437    79     1.59    0.1151    0.05   -0.3350    3.0238  -0.00008
 c_41        0.02964    0.2624    79     0.11    0.9104    0.05   -0.4926    0.5519  -0.00007
 c_42       -0.07768    0.1417    79    -0.55    0.5852    0.05   -0.3598    0.2044  -0.00046
 c_43        -0.4876    0.3797    79    -1.28    0.2028    0.05   -1.2434    0.2682  -0.00008
 v_u          0.6437   0.05278    79    12.19    <.0001    0.05    0.5386    0.7487  0.000229
*/


*Approximation;
PROC NLMIXED DATA = height_long;
	traject = x_1i + x_2i * age/12 + x_3i * (exp(beta_4*(age/12))-1) + x_4i*(beta_3*(age/12)*exp(beta_4*(age/12)));

	MODEL hght ~ NORMAL(traject, v_u);
	RANDOM x_1i x_2i x_3i x_4i ~ NORMAL([beta_1,beta_2,beta_3,0], [v_1,
									                        c_21,  v_2,
                                                            c_31, c_32,  v_3,
                                                            c_41, c_42, c_43, v_4])
	SUBJECT = id OUT = jb_estimates_tse;
	PARMS
        beta_1 = 51   beta_2 = 9     beta_3 = -18   beta_4 = -2
        v_1 = 8     v_2 = .8     v_3 = 12     v_4 = .2
        c_21 = .6   c_31 = 4     c_32 = 1     c_41 = 0     c_42 = 0     c_43 = 0
        v_u = .7;
RUN; 

/*
                                        Fit Statistics

                           -2 Log Likelihood                 1975.0
                           AIC (smaller is better)           2005.0
                           AICC (smaller is better)          2005.8
                           BIC (smaller is better)           2041.3

                                     Parameter Estimates

                      Standard
 Parameter  Estimate     Error    DF  t Value  Pr > |t|   Alpha     Lower     Upper  Gradient

 beta_1      51.0742    0.3529    79   144.74    <.0001    0.05   50.3719   51.7766  -0.01413
 beta_2       9.2963    0.1739    79    53.47    <.0001    0.05    8.9503    9.6424  0.004992
 beta_3     -17.8343    0.4923    79   -36.22    <.0001    0.05  -18.8142  -16.8543  0.011694
 beta_4      -2.0770   0.08449    79   -24.58    <.0001    0.05   -2.2451   -1.9088  0.025261
 v_1          7.4611    1.5321    79     4.87    <.0001    0.05    4.4114   10.5107   0.00498
 v_2          0.8125    0.3409    79     2.38    0.0196    0.05    0.1340    1.4910  0.009715
 v_3         11.7443    3.0483    79     3.85    0.0002    0.05    5.6767   17.8118  0.004387
 v_4         0.07635   0.07931    79     0.96    0.3387    0.05  -0.08152    0.2342  0.065215
 c_21         0.6530    0.5310    79     1.23    0.2225    0.05   -0.4040    1.7100  -0.00455
 c_31         4.5326    1.5953    79     2.84    0.0057    0.05    1.3572    7.7080  0.029762
 c_32         1.3652    0.8563    79     1.59    0.1149    0.05   -0.3393    3.0696   0.00341
 c_41        0.02678    0.2636    79     0.10    0.9194    0.05   -0.4980    0.5515  0.029458
 c_42       -0.07853    0.1437    79    -0.55    0.5862    0.05   -0.3645    0.2074  0.084053
 c_43        -0.4925    0.3857    79    -1.28    0.2054    0.05   -1.2601    0.2752  -0.02275
 v_u          0.6432   0.05280    79    12.18    <.0001    0.05    0.5381    0.7483  -0.04578
*/

PROC SORT DATA = jb_estimates_tse;
	BY id;
RUN;
DATA jb_estimates_tse_x1;
	SET jb_estimates_tse;
	IF effect = 'x_1i';
	x_1i = estimate;
	KEEP id x_1i;
RUN;
DATA jb_estimates_tse_x2;
	SET jb_estimates_tse;
	IF effect = 'x_2i';
	x_2i = estimate;
	KEEP id x_2i;
RUN;
DATA jb_estimates_tse_x3;
	SET jb_estimates_tse;
	IF effect = 'x_3i';
	x_3i = estimate;
	KEEP id x_3i;
RUN;
DATA jb_estimates_tse_x4;
	SET jb_estimates_tse;
	IF effect = 'x_4i';
	x_4i = estimate;
	KEEP id x_4i;
RUN;
DATA jb_estimates_tse1;
	MERGE 	jb_estimates_tse_x1 
			jb_estimates_tse_x2
			jb_estimates_tse_x3
			jb_estimates_tse_x4;
	BY id;
RUN;
 
DATA igs_prediction_tse;
	SET jb_estimates_tse1;
	beta_3 = -17.8343;
	beta_4 = -2.0770;

	DO age = 1 to 36 by .5;
		pred = x_1i + x_2i * age/12 + x_3i * (exp(beta_4*(age/12))-1) + x_4i*(beta_3*(age/12)*exp(beta_4*(age/12)));
	OUTPUT;
	END;
RUN;


PROC GPLOT DATA = igs_prediction_tse;
	TITLE1 'IGS Height by Age - Predictions';
	SYMBOL
		Interpol=join
		Color=black
		Repeat=5000
		Value=none
		Height=.5
		Width=1;
	AXIS1
		LABEL = (F=ARIAL H = 2 A=90 'Height (cm)')
		ORDER = (40 to 110 BY 10)
		MINOR = none;
	AXIS2
		LABEL = (F=ARIAL H = 2 'Age in Months')	
		ORDER = (0 to 40 BY 10)
		OFFSET= (5 PCT, 5 PCT)
		MINOR = none;

	PLOT pred * age = id/NOLEGEND VAXIS= AXIS1 HAXIS=AXIS2;
RUN;





* Spline Model with Variation in Knot Point;
PROC NLMIXED DATA = height_long QPOINTS = 5;
	traject = b_1i + b_2i*(MIN(age-b_4i,0)) + b_3i*(MAX(age-b_4i,0));

	MODEL hght ~ NORMAL(traject, v_u);
	RANDOM b_1i b_2i b_3i b_4i ~ NORMAL([beta_1,beta_2,beta_3,beta_4], [v_1,
									                           c_21,  v_2,
                                                               c_31, c_32,  v_3,
                                                               c_41, c_42, c_43, v_4])
	SUBJECT = id;
	PARMS
        beta_1 = 72   beta_2 = 2.5   beta_3 = .9    beta_4 = 7.7 
        v_1 = 7.4   v_2 = .08    v_3 = .003   v_4 = .09
        c_21 = 0    c_31 = 0     c_32 = 0     c_41 = 0    c_42 = 0    c_43 = 0
        v_u = 1.5; 
RUN;


/*
No Convergence
*/


* Spline Model with Variation in Knot Point;
PROC NLMIXED DATA = height_long QPOINTS = 1;
	traject = b_1i + b_2i*(MIN(age-b_4i,0)) + b_3i*(MAX(age-b_4i,0));

	MODEL hght ~ NORMAL(traject, v_u);
	RANDOM b_1i b_2i b_3i b_4i ~ NORMAL([beta_1,beta_2,beta_3,beta_4], [v_1,
									                           c_21,  v_2,
                                                               c_31, c_32,  v_3,
                                                                  0,    0,    0, v_4])
	SUBJECT = id;
	PARMS
        beta_1 = 72   beta_2 = 2.5   beta_3 = .9    beta_4 = 7.7 
        v_1 = 7.4   v_2 = .08    v_3 = .003   v_4 = .09
        c_21 = 0    c_31 = 0     c_32 = 0     
        v_u = 1.5; 
RUN;


PROC NLMIXED DATA = height_long QPOINTS = 1;
	traject = b_1i + b_2i*(MIN(age-b_4i,0)) + b_3i*(MAX(age-b_4i,0));

	MODEL hght ~ NORMAL(traject, v_u);
	RANDOM b_1i b_2i b_3i b_4i ~ NORMAL([beta_1,beta_2,beta_3,beta_4], [v_1,
									                           c_21,  v_2,
                                                               c_31, c_32,  v_3,
                                                               c_41, c_42, c_43, v_4])
	SUBJECT = id;
	PARMS
        beta_1 = 71   beta_2 = 2.5   beta_3 = .9    beta_4 = 7.5 
        v_1 = 8   v_2 = .1    v_3 = .1   v_4 = .1
        c_21 = 0    c_31 = 0     c_32 = 0     c_41 = 0    c_42 = 0    c_43 = 0
        v_u = 1.4; 
RUN;


                               beta_1        71.0111           1
                               beta_2         2.5433           1
                               beta_3         0.9032           1
                               beta_4         7.5006           1
                               v_1            7.9975           1
                               v_2           0.08665           1
                               v_3          0.003556           1
                               v_4           0.04812           1
                               c_21          0.04331           1
                               c_31          0.08042           1
                               c_32         0.006862           1
                               c_41          0.02653           1
                               c_42         -0.06044           1
                               c_43         -0.00069           1
                               v_u            1.4135           1

PROC NLMIXED DATA = height_long QPOINTS = 1;
	traject = b_1i + b_2i*(MIN(age-b_4i,0)) + b_3i*(MAX(age-b_4i,0));

	MODEL hght ~ NORMAL(traject, v_u);
	RANDOM b_1i b_2i b_3i b_4i ~ NORMAL([71.0111,2.5433,0.9032,7.5006], [7.9975,
									                           0.04331,  0.08665,
                                                               0.08042, 0.006862,  0.003556,
                                                               0.02653, -0.06044, -0.00069, 0.04812])
	SUBJECT = id;
	PARMS
        v_u = 1.4; 
RUN;

PROC NLMIXED DATA = height_long QPOINTS = 5;
	traject = b_1i + b_2i*(MIN(age-b_4i,0)) + b_3i*(MAX(age-b_4i,0));

	MODEL hght ~ NORMAL(traject, v_u);
	RANDOM b_1i b_2i b_3i b_4i ~ NORMAL([beta_1,beta_2,beta_3,beta_4], [7.9975,
									                           0.04331,  0.08665,
                                                               0.08042, 0.006862,  0.003556,
                                                               0.02653, -0.06044, -0.00069, 0.04812])
	SUBJECT = id;
	PARMS
        beta_1 = 71   beta_2 = 2.5   beta_3 = .9    beta_4 = 7.5 
        v_u = 1.4; 
RUN;

/*
                                     Parameter Estimates

                      Standard
 Parameter  Estimate     Error    DF  t Value  Pr > |t|   Alpha     Lower     Upper  Gradient

 beta_1      71.7426    0.3663    79   195.85    <.0001    0.05   71.0135   72.4717  -2.76E-6
 beta_2       2.5543   0.05468    79    46.71    <.0001    0.05    2.4455    2.6631  -8.45E-6
 beta_3       0.9032   0.01010    79    89.42    <.0001    0.05    0.8831    0.9233   0.00032
 beta_4       7.5716    0.1342    79    56.43    <.0001    0.05    7.3045    7.8387  0.000015
 v_u          1.4719    0.1039    79    14.17    <.0001    0.05    1.2652    1.6786  -0.00001
*/


PROC NLMIXED DATA = height_long QPOINTS = 1;
	traject = b_1i + b_2i*(MIN(age-b_4i,0)) + b_3i*(MAX(age-b_4i,0));

	MODEL hght ~ NORMAL(traject, v_u);
	RANDOM b_1i b_2i b_3i b_4i ~ NORMAL([beta_1,beta_2,beta_3,beta_4], [v_1,
									                           0,  v_2,
                                                               0, 0,  v_3,
                                                               0, 0, 0, .1])
	SUBJECT = id;
	PARMS
        beta_1 = 71   beta_2 = 2.5   beta_3 = .9    beta_4 = 7.5 
        v_1 = 8   v_2 = .1    v_3 = .1   
        v_u = 1.4; 
RUN;


PROC NLMIXED DATA = height_long QPOINTS = 1;
	traject = b_1i + b_2i*(MIN(age-b_4i,0)) + b_3i*(MAX(age-b_4i,0));

	MODEL hght ~ NORMAL(traject, v_u);
	RANDOM b_1i b_2i b_3i b_4i ~ NORMAL([beta_1,beta_2,beta_3,beta_4], [v_1,
									                           c_21,  v_2,
                                                               c_31, c_32,  v_3,
                                                               0, 0, 0, v_4])
	SUBJECT = id;
	PARMS
        beta_1 = 71   beta_2 = 2.5   beta_3 = .9    beta_4 = 7.5 
        v_1 = 8   v_2 = .1    v_3 = .1   v_4 = .05
        c_21 = 0    c_31 = 0     c_32 = 0     
        v_u = 1.4; 
RUN;


PROC NLMIXED DATA = height_long METHOD=FIRO;
	traject = b_1i + b_2i*(MIN(age-b_4i,0)) + b_3i*(MAX(age-b_4i,0));

	MODEL hght ~ NORMAL(traject, v_u);
	RANDOM b_1i b_2i b_3i b_4i ~ NORMAL([beta_1,beta_2,beta_3,beta_4], [v_1,
									                           c_21,  v_2,
                                                               c_31, c_32,  v_3,
                                                               c_41, c_42, c_43, v_4])
	SUBJECT = id;
	PARMS
        beta_1 = 72   beta_2 = 2.5   beta_3 = .9    beta_4 = 7.7 
        v_1 = 7.4   v_2 = .08    v_3 = .003   v_4 = .09
        c_21 = 0    c_31 = 0     c_32 = 0     c_41 = 0    c_42 = 0    c_43 = 0
        v_u = 1.5; 
RUN;

/*
                                        Fit Statistics

                           -2 Log Likelihood                 2244.8
                           AIC (smaller is better)           2274.8
                           AICC (smaller is better)          2275.6
                           BIC (smaller is better)           2311.1


                                     Parameter Estimates

                      Standard
 Parameter  Estimate     Error    DF  t Value  Pr > |t|   Alpha     Lower     Upper  Gradient

 beta_1      71.6981    0.3601    79   199.13    <.0001    0.05   70.9814   72.4148  -0.03371
 beta_2       2.5737   0.04600    79    55.95    <.0001    0.05    2.4821    2.6652   -0.0177
 beta_3       0.9016  0.009640    79    93.52    <.0001    0.05    0.8824    0.9207  0.002719
 beta_4       7.5130   0.09384    79    80.06    <.0001    0.05    7.3262    7.6998  0.051344
 v_1          7.3924    2.0628    79     3.58    0.0006    0.05    3.2865   11.4983  0.650156
 v_2        0.009574   0.02693    79     0.36    0.7231    0.05  -0.04402   0.06317  -0.08008
 v_3        0.002476  0.001126    79     2.20    0.0307    0.05  0.000236  0.004717  -1.17501
 v_4         -0.7295    0.1600    79    -4.56    <.0001    0.05   -1.0479   -0.4111  0.032705
 c_21         0.4529    0.1683    79     2.69    0.0087    0.05    0.1179    0.7880  0.046017
 c_31         0.1052   0.03445    79     3.06    0.0031    0.05   0.03667    0.1738  -0.09136
 c_32       0.002710  0.003885    79     0.70    0.4875    0.05  -0.00502   0.01044  -1.03104
 c_41        -0.3138    0.4251    79    -0.74    0.4626    0.05   -1.1599    0.5323  0.030849
 c_42         0.2108   0.04973    79     4.24    <.0001    0.05    0.1119    0.3098  0.020745
 c_43        0.02280  0.008144    79     2.80    0.0064    0.05  0.006593   0.03901  0.508375
 v_u          1.6367    0.1324    79    12.36    <.0001    0.05    1.3730    1.9003  -0.07212
*/

PROC NLMIXED DATA = height_long QPOINTS = 1;
	traject = b_1i + b_2i*(MIN(age-b_4i,0)) + b_3i*(MAX(age-b_4i,0));

	MODEL hght ~ NORMAL(traject, v_u);
	RANDOM b_1i b_2i b_3i b_4i ~ NORMAL([beta_1,beta_2,beta_3,beta_4], [v_1,
									                           c_21,  v_2,
                                                               c_31, c_32,  v_3,
                                                               c_41, c_42, c_43, v_4])
	SUBJECT = id;
	PARMS
        beta_1 = 72   beta_2 = 2.5   beta_3 = .9    beta_4 = 7.7 
        v_1 = 7.4   v_2 = .08    v_3 = .003     v_4 = .09
        c_21 = 0    c_31 = 0     c_32 = 0      c_41 = 0    c_42 = 0    c_43 = 0
        v_u = 1.5; 
RUN;

/*
                                        Fit Statistics

                           -2 Log Likelihood                 2279.0
                           AIC (smaller is better)           2309.0
                           AICC (smaller is better)          2309.8
                           BIC (smaller is better)           2345.2


                                     Parameter Estimates

                      Standard
 Parameter  Estimate     Error    DF  t Value  Pr > |t|   Alpha     Lower     Upper  Gradient

 beta_0      71.9987    3.9835    79    18.07    <.0001    0.05   64.0698   79.9276  1.154916
 beta_1       2.5082    0.7224    79     3.47    0.0008    0.05    1.0704    3.9460  -4.35429
 beta_2       0.9042  0.001991    79   454.11    <.0001    0.05    0.9002    0.9081  36.05993
 beta_3       7.6955  0.001392    79  5530.30    <.0001    0.05    7.6928    7.6983   6.64344
 v_0          7.3989   0.04850    79   152.57    <.0001    0.05    7.3024    7.4954  1.510214
 v_1         0.08111    1.2630    79     0.06    0.9490    0.05   -2.4329    2.5951  33.87268
 v_2        0.004197  0.000984    79     4.26    <.0001    0.05  0.002238  0.006157  258.9206
 v_3         0.06360  0.007584    79     8.39    <.0001    0.05   0.04850   0.07869  29.28242
 c_10        0.02161   12.2523    79     0.00    0.9986    0.05  -24.3659   24.4092  -22.7378
 c_20        0.05844    0.4368    79     0.13    0.8939    0.05   -0.8109    0.9278  -21.9736
 c_21        0.01182   0.01843    79     0.64    0.5231    0.05  -0.02486   0.04850  43.05608
 c_30        0.01229  0.008836    79     1.39    0.1683    0.05  -0.00530   0.02987  -13.1269
 c_31       -0.03784    0.2519    79    -0.15    0.8810    0.05   -0.5392    0.4636  58.75445
 c_32       -0.01507  0.000315    79   -47.92    <.0001    0.05  -0.01570  -0.01445  -23.2263
 v_u          1.4990  0.004736    79   316.49    <.0001    0.05    1.4895    1.5084  1.586492
*/


PROC NLMIXED DATA = height_long METHOD=ISAMP QPOINTS=1;
	traject = b_1i + b_2i*(MIN(age-b_4i,0)) + b_3i*(MAX(age-b_4i,0));

	MODEL hght ~ NORMAL(traject, v_u);
	RANDOM b_1i b_2i b_3i b_4i ~ NORMAL([beta_1,beta_2,beta_3,beta_4], [v_1,
									                           c_21,  v_2,
                                                               c_31, c_32,  v_3,
                                                               c_41, c_42, c_43, v_4])
	SUBJECT = id;
	PARMS
        beta_1 = 72   beta_2 = 2.5   beta_3 = .9    beta_4 = 7.7 
        v_1 = 7.4   v_2 = .1     v_3 = .003     v_4 = .09
        c_21 = 0    c_31 = 0     c_32 = 0      c_41 = 0    c_42 = 0    c_43 = 0
        v_u = 1.5; 
RUN;



PROC NLMIXED DATA = height_long QPOINTS = 1;
	traject = b_1i + b_2i*(MIN(age-b_4i,0)) + b_3i*(MAX(age-b_4i,0));

	MODEL hght ~ NORMAL(traject, s_u*s_u);
	RANDOM b_1i b_2i b_3i b_4i ~ NORMAL([beta_1,beta_2,beta_3,beta_4], [s_1*s_1,
									                           s_2*r_21*s_1,  s_2*s_2,
                                                               s_3*r_31*s_1, s_3*r_32*s_2,  s_3*s_3,
                                                               0, 0, 0, s_4*s_4])
	SUBJECT = id;
	PARMS
        beta_1 = 71.9   beta_2 = 2.51   beta_3 = .9    beta_4 = 7.7 
        s_1 = 2.7   s_2 = .29     s_3 = .06      s_4 = .30
        r_21 = .1    r_31 = .5     r_32 = .4     
        s_u = 1.21; 
RUN;

/*
                                       Fit Statistics

                           -2 Log Likelihood                 2264.7
                           AIC (smaller is better)           2288.7
                           AICC (smaller is better)          2289.2
                           BIC (smaller is better)           2317.7


                                     Parameter Estimates

                      Standard
 Parameter  Estimate     Error    DF  t Value  Pr > |t|   Alpha     Lower     Upper  Gradient

 beta_1      71.6888    0.3332    79   215.14    <.0001    0.05   71.0255   72.3520  0.001165
 beta_2       2.5651   0.05275    79    48.63    <.0001    0.05    2.4601    2.6701  -0.03237
 beta_3       0.9031  0.009926    79    90.99    <.0001    0.05    0.8833    0.9229  0.197818
 beta_4       7.5172    0.1331    79    56.48    <.0001    0.05    7.2522    7.7821  0.017235
 s_1          2.4324    0.2155    79    11.29    <.0001    0.05    2.0035    2.8613  0.024373
 s_2          0.2640   0.03882    79     6.80    <.0001    0.05    0.1868    0.3413  0.080178
 s_3         0.05719  0.009823    79     5.82    <.0001    0.05   0.03764   0.07674  0.415748
 s_4        -0.00077    0.1369    79    -0.01    0.9955    0.05   -0.2732    0.2716   -0.0413
 r_21         0.4154    0.1307    79     3.18    0.0021    0.05    0.1553    0.6755  0.054432
 r_31         0.5456    0.1710    79     3.19    0.0020    0.05    0.2052    0.8860  0.033984
 r_32         0.5455    0.2436    79     2.24    0.0280    0.05   0.06057    1.0305  0.041148
 s_u          1.2150   0.04543    79    26.74    <.0001    0.05    1.1246    1.3055  0.048704
*/


PROC NLMIXED DATA = height_long QPOINTS = 1;
	traject = b_1i + b_2i*(MIN(age-b_4i,0)) + b_3i*(MAX(age-b_4i,0));

	MODEL hght ~ NORMAL(traject, s_u*s_u);
	RANDOM b_1i b_2i b_3i b_4i ~ NORMAL([beta_1,beta_2,beta_3,beta_4], [s_1*s_1,
									                           s_2*r_21*s_1,  s_2*s_2,
                                                               s_3*r_31*s_1, s_3*r_32*s_2,  s_3*s_3,
                                                               0, 0, 0, s_4*s_4])
	SUBJECT = id OUT=sp_estimates;
	PARMS
        beta_1 = 71.9   beta_2 = 2.51   beta_3 = .9    beta_4 = 7.9 
        s_1 = 2.7   s_2 = .29     s_3 = .06      s_4 = .30
        r_21 = .1    r_31 = .5     r_32 = .4     
        s_u = 1.21; 
RUN;

/*
                                        Fit Statistics

                           -2 Log Likelihood                 2264.7
                           AIC (smaller is better)           2288.7
                           AICC (smaller is better)          2289.2
                           BIC (smaller is better)           2317.7


                                     Parameter Estimates

                      Standard
 Parameter  Estimate     Error    DF  t Value  Pr > |t|   Alpha     Lower     Upper  Gradient

 beta_1      71.6883    0.3331    79   215.21    <.0001    0.05   71.0252   72.3513  0.004454
 beta_2       2.5653   0.05273    79    48.65    <.0001    0.05    2.4603    2.6702  -0.03143
 beta_3       0.9031  0.009922    79    91.02    <.0001    0.05    0.8833    0.9228  -0.08793
 beta_4       7.5167    0.1330    79    56.50    <.0001    0.05    7.2519    7.7815  -0.02099
 s_1          2.4315    0.2153    79    11.29    <.0001    0.05    2.0030    2.8601  0.015766
 s_2          0.2638   0.03880    79     6.80    <.0001    0.05    0.1866    0.3410   -0.0216
 s_3         0.05715  0.009817    79     5.82    <.0001    0.05   0.03761   0.07669  -0.09636
 s_4        0.001486    0.1368    79     0.01    0.9914    0.05   -0.2708    0.2738  0.079414
 r_21         0.4147    0.1308    79     3.17    0.0022    0.05    0.1544    0.6750   0.02536
 r_31         0.5451    0.1711    79     3.19    0.0021    0.05    0.2046    0.8856  -0.00105
 r_32         0.5463    0.2437    79     2.24    0.0278    0.05   0.06119    1.0313  0.060836
 s_u          1.2149   0.04542    79    26.75    <.0001    0.05    1.1245    1.3053  -0.05472
*/

*Predicted Plot based on OUT=estimate;
PROC SORT DATA = sp_estimates;
	BY id;
RUN;
DATA sp_estimates_b1;
	SET sp_estimates;
	IF effect = 'b_1i';
	b_1i = estimate;
	KEEP id b_1i;
RUN;
DATA sp_estimates_b2;
	SET sp_estimates;
	IF effect = 'b_2i';
	b_2i = estimate;
	KEEP id b_2i;
RUN;
DATA sp_estimates_b3;
	SET sp_estimates;
	IF effect = 'b_3i';
	b_3i = estimate;
	KEEP id b_3i;
RUN;
DATA sp_estimates_b4;
	SET sp_estimates;
	IF effect = 'b_4i';
	b_4i = estimate;
	KEEP id b_4i;
RUN;
DATA sp_estimates1;
	MERGE 	sp_estimates_b1 
			sp_estimates_b2 
			sp_estimates_b3
			sp_estimates_b4;
	BY id;
RUN;
 
*Creating a longitudinal database with predictions;
DATA igs_prediction;
	MERGE height_long sp_estimates1;
	BY id;
	pred = b_1i + b_2i*(MIN(age-b_4i,0)) + b_3i*(MAX(age-b_4i,0));
	resid = hght - pred;
RUN;

PROC GPLOT DATA = igs_prediction;
	TITLE1 'IGS Height by Age - Predictions';
	SYMBOL
		Interpol=join
		Color=black
		Repeat=5000
		Value=dot
		Height=.5
		Width=1;
	AXIS1
		LABEL = (F=ARIAL H = 2 A=90 'Height (cm)')
		ORDER = (40 to 110 BY 10)
		OFFSET= (5 PCT, 5 PCT)
		MINOR = none;
	AXIS2
		LABEL = (F=ARIAL H = 2 'Age in Months')	
		ORDER = (0 to 40 BY 10)
		OFFSET= (5 PCT, 5 PCT)
		MINOR = none;

	PLOT pred * age = id/NOLEGEND VAXIS= AXIS1 HAXIS=AXIS2 HREF=7.5167;
RUN;



DATA igs_prediction;
	SET sp_estimates1;
	DO age = 1 to 36 by .01;
	pred = b_1i + b_2i*(MIN(age-b_4i,0)) + b_3i*(MAX(age-b_4i,0));
		OUTPUT;
	END;
RUN;

PROC GPLOT DATA = igs_prediction;
	TITLE1 'IGS Height by Age - Predictions';
	SYMBOL
		Interpol=join
		Color=black
		Repeat=5000
		Value=none
		Height=.5
		Width=1;
	AXIS1
		LABEL = (F=ARIAL H = 2 A=90 'Height (cm)')
		ORDER = (40 to 110 BY 10)
		OFFSET= (5 PCT, 5 PCT)
		MINOR = none;
	AXIS2
		LABEL = (F=ARIAL H = 2 'Age in Months')	
		ORDER = (0 to 40 BY 10)
		OFFSET= (5 PCT, 5 PCT)
		MINOR = none;

	PLOT pred * age = id/NOLEGEND VAXIS= AXIS1 HAXIS=AXIS2 HREF=7.5167;
RUN;


*Without Variation;
PROC NLMIXED DATA = height_long QPOINTS = 1;
	traject = b_1i + b_2i*(MIN(age-gamma,0)) + b_3i*(MAX(age-gamma,0));

	MODEL hght ~ NORMAL(traject, s_u*s_u);
	RANDOM b_1i b_2i b_3i ~ NORMAL([beta_1,beta_2,beta_3], [s_1*s_1,
									                           s_2*r_21*s_1,  s_2*s_2,
                                                               s_3*r_31*s_1, s_3*r_32*s_2,  s_3*s_3])
	SUBJECT = id;
	PARMS
        beta_1 = 71.9   beta_2 = 2.51   beta_3 = .9    gamma = 7.9 
        s_1 = 2.7   s_2 = .29     s_3 = .06      
        r_21 = .1    r_31 = .5     r_32 = .4     
        s_u = 1.21; 
RUN;


/*
                                       Fit Statistics

                           -2 Log Likelihood                 2264.7
                           AIC (smaller is better)           2286.7
                           AICC (smaller is better)          2287.2
                           BIC (smaller is better)           2313.3


                                     Parameter Estimates

                      Standard
 Parameter  Estimate     Error    DF  t Value  Pr > |t|   Alpha     Lower     Upper  Gradient

 beta_1      71.6876    0.3330    80   215.26    <.0001    0.05   71.0248   72.3503  -0.00864
 beta_2       2.5651   0.05274    80    48.64    <.0001    0.05    2.4602    2.6701  -0.03516
 beta_3       0.9031  0.009931    80    90.94    <.0001    0.05    0.8833    0.9229  0.228625
 gamma        7.5169    0.1331    80    56.49    <.0001    0.05    7.2521    7.7817  0.018196
 s_1          2.4305    0.2151    80    11.30    <.0001    0.05    2.0024    2.8586   -0.0156
 s_2          0.2639   0.03879    80     6.80    <.0001    0.05    0.1867    0.3411  0.008054
 s_3         0.05725  0.009820    80     5.83    <.0001    0.05   0.03771   0.07680  0.775556
 r_21         0.4149    0.1307    80     3.18    0.0021    0.05    0.1548    0.6749  0.057159
 r_31         0.5441    0.1710    80     3.18    0.0021    0.05    0.2038    0.8844  -0.00004
 r_32         0.5435    0.2436    80     2.23    0.0285    0.05   0.05874    1.0282  0.021639
 s_u          1.2149   0.04541    80    26.75    <.0001    0.05    1.1245    1.3052  0.013988
*/




PROC NLMIXED DATA = height_long METHOD = FIRO;
	traject = b_1i + b_2i*(MIN(age-b_4i,0)) + b_3i*(MAX(age-b_4i,0));

	MODEL hght ~ NORMAL(traject, v_u);
	RANDOM b_1i b_2i b_3i b_4i ~ NORMAL([beta_1,beta_2,beta_3,beta_4], [v_1,
									                           c_21,  v_2,
                                                               c_31, c_32,  v_3,
                                                               c_41, c_42, c_43, v_4])
	SUBJECT = id;
	PARMS	
		beta_1 = 72	beta_2 = 2.5 	beta_3 = .9	beta_4 = 7.7
	
		v_1 = 7.4	v_2 = .08	v_3 = .003  v_4 = .09
		c_21 = 0	c_31 = 0	c_32 = 0   c_41 = 0   c_42 = 0   c_43 = 0

		v_u = 1.5;
RUN;

/*


                                        Fit Statistics

                           -2 Log Likelihood                 2244.8
                           AIC (smaller is better)           2274.8
                           AICC (smaller is better)          2275.6
                           BIC (smaller is better)           2311.1


                                     Parameter Estimates

                      Standard
 Parameter  Estimate     Error    DF  t Value  Pr > |t|   Alpha     Lower     Upper  Gradient

 beta_0      71.6981    0.3601    79   199.13    <.0001    0.05   70.9814   72.4148  -0.03371
 beta_1       2.5737   0.04600    79    55.95    <.0001    0.05    2.4821    2.6652   -0.0177
 beta_2       0.9016  0.009640    79    93.52    <.0001    0.05    0.8824    0.9207  0.002719
 beta_3       7.5130   0.09384    79    80.06    <.0001    0.05    7.3262    7.6998  0.051344
 v_0          7.3924    2.0628    79     3.58    0.0006    0.05    3.2865   11.4983  0.650156
 v_1        0.009574   0.02693    79     0.36    0.7231    0.05  -0.04402   0.06317  -0.08008
 v_2        0.002476  0.001126    79     2.20    0.0307    0.05  0.000236  0.004717  -1.17501
 v_3         -0.7295    0.1600    79    -4.56    <.0001    0.05   -1.0479   -0.4111  0.032705
 c_10         0.4529    0.1683    79     2.69    0.0087    0.05    0.1179    0.7880  0.046017
 c_20         0.1052   0.03445    79     3.06    0.0031    0.05   0.03667    0.1738  -0.09136
 c_21       0.002710  0.003885    79     0.70    0.4875    0.05  -0.00502   0.01044  -1.03104
 c_30        -0.3138    0.4251    79    -0.74    0.4626    0.05   -1.1599    0.5323  0.030849
 c_31         0.2108   0.04973    79     4.24    <.0001    0.05    0.1119    0.3098  0.020745
 c_32        0.02280  0.008144    79     2.80    0.0064    0.05  0.006593   0.03901  0.508375
 v_u          1.6367    0.1324    79    12.36    <.0001    0.05    1.3730    1.9003  -0.07212
*/







PROC NLMIXED DATA = height_long QPOINTS=1;
	traject = x_1i + x_2i * (age-beta_4) + x_3i * ABS(age-beta_4) + x_4i*(-beta_2*ABS(age-beta_4)+beta_3*age-beta_3*beta_4)/ABS(age-beta_4);

	MODEL hght ~ NORMAL(traject, v_u);
	RANDOM x_1i x_2i x_3i x_4i ~ NORMAL([beta_1,beta_2,beta_3,0], [v_1,
									                        c_21,  v_2,
                                                            c_31, c_32,  v_3,
                                                            c_41, c_42, c_43, v_4])
	SUBJECT = id;
	PARMS	
		beta_1 = 72	beta_2 = 2.5 	beta_3 = -.9	beta_4 = 7.5
	
		v_1 = 7.4	v_2 = .08	v_3 = .003  v_4 = .09
		c_21 = 0	c_31 = 0	c_32 = 0   c_41 = 0   c_42 = 0   c_43 = 0

		v_u = 1.5;
RUN;

